/**
 * jquery.textarea-highlighter.js - jQuery plugin for highlighting text in textarea.
 * @version v0.6.3
 * @link https://github.com/marexandre/jquery.textarea-highlighter.js
 * @author alexandre.kirillov@gmail.com
 * @license MIT license. http://opensource.org/licenses/MIT
 */
(function (factory) {
  // UMD[2] wrapper for jQuery plugins to work in AMD or in CommonJS.
  //
  // [2] https://github.com/umdjs/umd

  if (typeof define === 'function' && define.amd) {
    // AMD. Register as an anonymous module.
    define(['jquery'], factory);
  } else if (typeof exports === 'object') {
    // Node/CommonJS
    module.exports = factory(require('jquery'));
  } else {
    // Browser globals
    factory(jQuery);
  }
}(function (jQuery) {
    var marexandre;
    (function (marexandre) {
      'use strict';

      var Helper = (function() {
        function Helper() {}

        Helper.prototype.orderBy = function(list, type) {
          return list.sort(function(a, b) {
            return parseInt(a[type], 10) - parseInt(b[type], 10);
          });
        };

        Helper.prototype.removeOverlapingIndecies = function(list) {
          var a = [], item, next;

          // Check for overlapping items
          for (var i = 0, imax = list.length; i < imax; i++) {
            item = list[i];

            for (var j = i + 1; j < imax; j++) {
              next = list[j];

              if (this.isOverlap(item, next)) {
                a.push(j);
              }
            }
          }
          // Remove overlapping items from the list
          return list.slice(0).filter(function(elem, pos) {
            if (a.indexOf(pos) !== -1) {
              return false;
            }
            return true;
          });
        };

        Helper.prototype.isOverlap = function(x, y) {
          return x.start < y.end && y.start < x.end;
        };

        Helper.prototype.cleanupOnWordBoundary = function(text, list, useWordBoundary) {
          useWordBoundary = useWordBoundary || true;

          var a = [], o, w, ww;

          for (var i = 0, imax = list.length; i < imax; i++) {
            o = list[i];
            w = text.slice(o.start, o.end);
            ww = text.slice(o.start - 1, o.end + 1);

            if (useWordBoundary && this.isWrappedByASCII(w) && !this.checkWordBoundary(w, ww)) {
              a.push(i);
            }
          }
          // Remove overlapping items from the list
          return list.slice(0).filter(function(elem, pos) {
            if (a.indexOf(pos) !== -1) {
              return false;
            }
            return true;
          });
        };

        Helper.prototype.makeTokenized = function(text, indecies) {
          var a = [], o, s = 0, ss = 0;

          for (var i = 0, imax = indecies.length; i < imax; i++) {
            o = indecies[i];
            if (o.end < o.start) {
              continue;
            }
            ss = o.start;

            if (ss > s) {
              a.push({ 'value': text.slice(s, ss), 'type': 'text' });
            }

            a.push({ 'value': text.slice(ss, o.end), 'type': o.type });

            s = o.end;
          }

          if (s < text.length) {
            a.push({ 'value': text.slice(s, text.length), 'type': 'text' });
          }

          return a;
        };

        Helper.prototype.checkWordBoundary = function(w, ww) {
          return new RegExp('\\b' + this.escapeRegExp(w) + '\\b').test(ww);
        };

        Helper.prototype.isWrappedByASCII = function(str) {
          return /^\w.*\w$|^\w+$/.test(str);
        };

        Helper.prototype.escapeHTML = function(str) {
          return str
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
        };

        Helper.prototype.escapeRegExp = function(str) {
          return str.replace(/[\-\[\]{}()*+?.,\\\^$|#\s]/g, '\\$&');
        };

        Helper.prototype.sanitizeBreakLines = function(str) {
          return str
            .replace(/\r\n/g, '\n')
            .replace(/\r/g, '\n');
        };

        Helper.prototype.getUniqueArray = function(a) {
          return a.filter(function(elem, pos, self) {
            if (elem === '') {
              return false;
            }
            return self.indexOf(elem) === pos;
          });
        };

        Helper.prototype.createHTML = function(tokenized) {
          var a = [];

          for (var i = 0, imax = tokenized.length; i < imax; i++) {
            if (tokenized[i].type === 'text') {
              a.push(tokenized[i].value);
            } else {
              a.push(this.getTextInSpan(tokenized[i].type, tokenized[i].value));
            }
          }

          return a.join('');
        };

        Helper.prototype.getTextInSpan = function(className, text) {
          return '<span class="' + className + '">' + text + '</span>';
        };

        Helper.prototype.browser = function() {
          var userAgent = navigator.userAgent,
              msie    = /(msie|trident)/i.test( userAgent ),
              chrome  = /chrome/i.test( userAgent ),
              firefox = /firefox/i.test( userAgent ),
              safari  = /safari/i.test( userAgent ) && !chrome,
              iphone  = /iphone/i.test( userAgent );

          if ( msie ) { return { msie: true }; }
          if ( chrome ) { return { chrome: true }; }
          if ( firefox ) { return { firefox: true }; }
          if ( iphone ) { return { iphone: true }; }
          if ( safari ) { return { safari: true }; }

          return {
            msie   : false,
            chrome : false,
            firefox: false,
            safari : false,
            iphone : false
          };
        };

        return Helper;
      })();

      marexandre.Helper = Helper;
    })(marexandre || (marexandre = {}));

    var marexandre;
    (function($, window, document, undefined) {
      'use strict';

      var pluginName = 'textareaHighlighter';
      var helper = new marexandre.Helper();

      var TextareaHighlighter = function($el, options) {
        this.$element  = $el;
        this.element   = this.$element[0];
        this.settings  = $.extend( {}, TextareaHighlighter.DEFAULTS, this.$element.data(), options );

        this.$wrapDiv       = $(document.createElement('div')).addClass('textarea-highlighter-wrap');
        this.$backgroundDiv = $(document.createElement('div')).addClass('background-div ' + this.$element.attr('class') );
        this.$autoSize      = $('<pre><div class="autosize"></div></pre>').addClass( this.$element.attr('class') ).hide();
        this.$autoSizeElement = this.$autoSize.find('.autosize');

        this.init();
      };

      TextareaHighlighter.DEFAULTS = {
        //word_base: true,
        matches: [
          // {'matchClass': '', 'match': []}
        ],
        isAutoExpand     : true,
        typingDelay      : 30,
        debug            : false
      };

      TextareaHighlighter.prototype.init = function() {
        var _this          = this,
            $this          = this.$element,
            settings       = this.settings,
            $wrapDiv       = this.$wrapDiv,
            $backgroundDiv = this.$backgroundDiv;

        _this.updateStyle();

        // insert backgroundDiv
        $this.wrap( $wrapDiv ).before( $backgroundDiv );
        // Insert auto resize div
        if (settings.isAutoExpand) {
          $this.after( _this.$autoSize );
        }

        _this.updateHeight();
        _this.bindEvents();
        _this.highlight();
      };

      TextareaHighlighter.prototype.bindEvents = function() {
        var _this = this;
        var $this = this.$element;

        $this
          .data('highlighterTimerId', -1)
          // Watch on scroll event
          .on('scroll.textarea.highlighter', function() {
            _this.$backgroundDiv.scrollTop( $this.scrollTop() );
          });

        if ('onpropertychange' in _this.element) {
          var lastUpdate = new Date().getTime();
          var timeDiff = 0;
          var abs = Math.abs;
          // IE 9+
          $this.on('input.textarea.highlighter keyup.textarea.highlighter', function(e) {
            timeDiff = abs(lastUpdate - new Date().getTime());

            if (timeDiff > 10) {
              _this.change(e);
              lastUpdate = new Date().getTime();
            }
          });
          // For backspace long press
          $this.on('keydown.textarea.highlighter', function(e) {
            timeDiff = abs(lastUpdate - new Date().getTime());

            if (e.which === 8 && (timeDiff < 10 || 250 < timeDiff)) {
              _this.change(e);
              lastUpdate = new Date().getTime();
            }
          });
        } else {
          // Modern browsers
          $this.on('input.textarea.highlighter', function(e) {
            _this.change(e);
          });
        }
      };

      TextareaHighlighter.prototype.change = function(e) {
        var _this = this;

        // if arrow keys, don't do anything
        if (/(37|38|39|40)/.test(e.keyCode)) {
          return true;
        }

        _this.updateHeight();

        // TODO: replace this stupid thing with proper 'throttle'
        // check for last update, this is for performace
        if (_this.$element.data('highlighterTimerId') !== -1) {
          clearTimeout( _this.$element.data('highlighterTimerId') );
          _this.$element.data('highlighterTimerId', -1);
        }

        // id for set timeout
        var changeId = setTimeout(function() {
          _this.highlight();
        }, _this.settings.typingDelay);
        // set setTimeout id
        _this.$element.data('highlighterTimerId', changeId);
      };

      TextareaHighlighter.prototype.highlight = function() {
        var _this = this;
        var text = _this.$element.val();
        var settings = _this.settings;

        // Escape HTML content
        //text = helper.escapeHTML(text);
        text = _this.getHighlightedContent(text);

        _this.$backgroundDiv.html(text);
        _this.updateHeight();
        _this.$element.trigger('textarea.highlighter.highlight');
      };

      TextareaHighlighter.prototype.getHighlightedContent = function(text) {
        var _this = this;
        var list = JSON.parse(JSON.stringify(_this.settings.matches));

        for (var i = 0, imax = list.length; i < imax; i++) {
            list[i].end++;
        }

        list = helper.orderBy(list, 'start');
        list = helper.removeOverlapingIndecies(list);
        //list = helper.cleanupOnWordBoundary(text, list, _this.settings.word_base);

        var tokens = helper.makeTokenized(text, list);
        for (var i = 0, imax = tokens.length; i < imax; i++) {
            tokens[i].value = helper.escapeHTML(tokens[i].value);
        }
        return helper.createHTML(tokens);
      };

      TextareaHighlighter.prototype.updateMatches = function(matches) {
        var _this = this;
        _this.settings.matches = matches;
        _this.highlight();
      };

      TextareaHighlighter.prototype.updateStyle = function() {
        var _this    = this;
        var $this    = this.$element;
        var settings = this.settings;
        var style = {
          paddingTop   : parseInt( $this.css('padding-top'), 10 ),
          paddingRight : parseInt( $this.css('padding-right'), 10 ),
          paddingBottom: parseInt( $this.css('padding-bottom'), 10 ),
          paddingLeft  : parseInt( $this.css('padding-left'), 10 )
        };

        // Hack for iPhone
        if (helper.browser().iphone) {
          style.paddingRight += 3;
          style.paddingLeft += 3;
        }

        // wrap div
        this.$wrapDiv.css({
          'position': 'relative'
        });

        // background div
        this.$backgroundDiv.css({
          'position'      : 'absolute',
          'height'        : '100%',
          'font-family'   : 'inherit',
          'color'         : ( settings.debug ) ? '#f00' : 'transparent',
          'padding-top'   : style.paddingTop,
          'padding-right' : style.paddingRight,
          'padding-bottom': style.paddingBottom,
          'padding-left'  : style.paddingLeft
        });
        _this.cloneCSSToTarget( _this.$backgroundDiv );

        if (settings.isAutoExpand) {
          // auto size div
          _this.$autoSize.css({
            'top'           : 0,
            'left'          : 0,
            'font-family'   : 'inherit',
            'position'      : 'absolute',
            'padding-top'   : style.paddingTop,
            'padding-right' : style.paddingRight,
            'padding-bottom': style.paddingBottom,
            'padding-left'  : style.paddingLeft
          });
          _this.cloneCSSToTarget( _this.$autoSize );
        }

        // text area element
        $this.css({
          'color'     : ( settings.debug ) ? 'rgba(0,0,0,0.5)' : 'inherit',
          'position'  : 'relative',
          'background': 'none'
        });
      };

      TextareaHighlighter.prototype.updateHeight = function() {
        var _this = this;

        if (_this.settings.isAutoExpand) {
          _this.$autoSizeElement.html(helper.escapeHTML( helper.sanitizeBreakLines(_this.$element.val()) ) + ' ');
          var h = _this.$autoSize.height();
          // If the height of textarea changed then update it
          if (_this.$element.height() !== h) {
            _this.$element.height(h);
            _this.$backgroundDiv.height(h);
          }
        }
      };

      TextareaHighlighter.prototype.cloneCSSToTarget = function($t) {
        var $element = this.$element;
        var cloneCSSProperties = [
          'lineHeight', 'textDecoration', 'letterSpacing',
          'fontSize', 'fontStyle',
          'fontWeight', 'textTransform', 'textAlign',
          'direction', 'wordSpacing', 'fontSizeAdjust',
          'wordWrap', 'word-break',
          'marginLeft', 'marginRight',
          'marginTop','marginBottom',
          'borderLeftWidth', 'borderRightWidth',
          'borderTopWidth','borderBottomWidth',
          'boxSizing', 'webkitBoxSizing', 'mozBoxSizing', 'msBoxSizing'
        ];
        var val = null;

        $.each(cloneCSSProperties, function(i, p) {
          val = $element.css(p);
          // Only set if different to prevent overriding percentage css values.
          if ($t.css(p) !== val) {
            $t.css(p, val);
          }
        });
      };

      TextareaHighlighter.prototype.destroy = function() {
        $.data( this.element, 'plugin_' + pluginName, false );
        this.$backgroundDiv.remove();
        this.$autoSize.remove();
        this.$element
          .data('highlighterTimerId', -1)
          // unbind all events
          .off('scroll.textarea.highlighter')
          .off('input.textarea.highlighter')
          .off('keyup.textarea.highlighter')
          .off('propertychange.textarea.highlighter')
          // reset all styles
          .attr('style', '')
          .unwrap();
      };

      TextareaHighlighter.prototype.debugModeOn = function() {
        this.settings.debug = true;
        this.$backgroundDiv.css({ 'color': '#f00' });
        this.$element.css({ 'color': 'rgba(0,0,0,0.5)' });
      };

      TextareaHighlighter.prototype.debugModeOff = function() {
        this.settings.debug = false;
        this.$backgroundDiv.css({ 'color': 'transparent' });
        this.$element.css({ 'color': 'inherit' });
      };

      $.fn.textareaHighlighter = function(option) {
        var args = arguments;

        return this.each(function() {
          var $this = $(this);
          var data = $this.data(pluginName);
          var options = typeof option === 'object' && option;

          // If no options or plugin was NOT initialized yet, do NOT do anything.
          if (!option || (!data && typeof option === 'string')) {
            return;
          }

          if (!data) {
            data = new TextareaHighlighter($this, options);
            $this.data(pluginName, data);
          }

          if (typeof option === 'string') {
            if (!data[option]) {
              throw 'Unknown method: ' + option;
            }

            data[option].apply(data, Array.prototype.slice.call(args, 1));
          }
        });
      };

    })(jQuery, window, document);
}));


